using System;
using System.IO;
using System.Runtime.InteropServices;
using DynaPDF;

namespace text_coordinates
{
   class TextCoordinates
   {
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      // This class marks the search text in the PDF file.
      private static CTextCoordinates m_TextCoords;

      static int  parseBeginTemplate(IntPtr Data, IntPtr Object, int Handle, ref TPDFRect BBox, IntPtr Matrix)
      {
         return m_TextCoords.BeginTemplate(BBox, Matrix);
      }

      static void parseEndTemplate(IntPtr Data)
      {
         m_TextCoords.EndTemplate();
      }

      static void parseMulMatrix(IntPtr Data, IntPtr Object, ref TCTM Matrix)
      {
         m_TextCoords.MulMatrix(Matrix);
      }

      static int parseRestoreGraphicState(IntPtr Data)
      {
         m_TextCoords.RestoreGState();
         return 0;
      }

      static int parseSaveGraphicState(IntPtr Data)
      {
         return m_TextCoords.SaveGState();
      }

      static void parseSetCharSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextCoords.SetCharSpacing(Value);
      }

      static void parseSetFont(IntPtr Data, IntPtr Object, TFontType Type, int Embedded, IntPtr FontName, TFStyle Style, double FontSize, IntPtr IFont)
      {
         m_TextCoords.SetFont(IFont, FontSize, Type);
      }

      static void parseSetTextDrawMode(IntPtr Data, IntPtr Object, TDrawMode Mode)
      {
         m_TextCoords.SetTextDrawMode(Mode);
      }

      static void parseSetTextScale(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextCoords.SetTextScale(Value);
      }

      static void parseSetWordSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextCoords.SetWordSpacing(Value);
      }

      static int parseShowTextArrayW(IntPtr Data, TTextRecordA[] Source, ref TCTM Matrix, TTextRecordW[] Kerning, int Count, double Width, int Decoded)
      {
         return m_TextCoords.MarkCoordinates(Matrix, Source, Kerning, Count, Width, Decoded != 0);
      }

      [STAThread]
      static void Main(string[] args)
      {
         try
         {
            /*
               This example draws lines under each text record of a PDF file to check whether the
               coordinate calculations are correct. To visualize how the text is stored in the PDF
               file the line color changes from blue to red or vice versa each time a new text
               record is processed.
            */
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // The output file wil be created later

            // External CMaps should always be loaded when processing text from PDF files.
            // See the description of ParseContent() for further information.
            pdf.SetCMapDir(System.IO.Path.GetFullPath("../../../../../Resource/CMap/"), DynaPDF.TLoadCMapFlags.lcmDelayed | DynaPDF.TLoadCMapFlags.lcmRecursive);

            // We avoid the conversion of pages to templates
            pdf.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage);
            if (pdf.OpenImportFile("../../../../../dynapdf_help.pdf", TPwdType.ptOpen, null) < 0)
            {
               Console.Write("Input file \"../../../../../../dynapdf_help.pdf\" not found!\n");
               Console.Read();
               return;
            }
            pdf.ImportPDFFile(1, 1.0, 1.0);
            pdf.CloseImportFile();

            // We flatten markup annotations and form fields so that we can process the text of these objects too.
            pdf.FlattenAnnots(TAnnotFlattenFlags.affMarkupAnnots);
            pdf.FlattenForm();

            TPDFParseInterface  stack = new TPDFParseInterface();
            stack.BeginTemplate       = new TBeginTemplate(parseBeginTemplate);
            stack.EndTemplate         = new TEndTemplate(parseEndTemplate);
            stack.MulMatrix           = new TMulMatrix(parseMulMatrix);
            stack.RestoreGraphicState = new TRestoreGraphicState(parseRestoreGraphicState);
            stack.SaveGraphicState    = new TSaveGraphicState(parseSaveGraphicState);
            stack.SetCharSpacing      = new TSetCharSpacing(parseSetCharSpacing);
            stack.SetFont             = new TSetFont(parseSetFont);
            stack.SetTextScale        = new TSetTextScale(parseSetTextScale);
            stack.SetWordSpacing      = new TSetWordSpacing(parseSetWordSpacing);
            stack.ShowTextArrayW      = new TShowTextArrayW(parseShowTextArrayW);

            m_TextCoords = new CTextCoordinates(pdf);
            int i, count = pdf.GetPageCount();
            for (i = 1; i <= count; i++)
            {
               pdf.EditPage(i);
               pdf.SetLineWidth(0.5);

               m_TextCoords.Init();
               pdf.ParseContent(IntPtr.Zero, ref stack, DynaPDF.TParseFlags.pfNone);
               pdf.EndPage();
            }
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }
         catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
      }
   }
}
